program NONLINEARSEIDEL;
{--------------------------------------------------------------------}
{  Alg2'9.pas   Pascal program for implementing Algorithm 2.9        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 2.9 (Nonlinear Seidel Iteration).                       }
{  Section   2.6, Iteration for Nonlinear Systems, Page 108          }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    FunMax = 6;
    Max = 99;
    Rmax = 101;
    Epsilon = 1E-7;

  type
    VECTOR = array[1..2] of real;
    LVECTR = array[0..Rmax] of real;
    LETTERS = string[200];
    STATUS = (Computing, Done, More, Working);

  var
    Count, FunType, Inum, Meth, Sub: integer;
    Rnum, Sep, Tol: real;
    G0, P0, P1, Pstart: VECTOR;
    VP, VQ: LVECTR;
    Ans: CHAR;
    Mess: LETTERS;
    Stat, State: STATUS;

  procedure FUN (P: VECTOR; var G: VECTOR);
    var
      X, Y: real;
  begin
    X := P[1];
    Y := P[2];
    case FunType of
      1: 
        begin
          G[1] := (X * X - Y + 0.5) / 2;
          if Meth = 2 then
            X := G[1];
          G[2] := (-X * X - 4 * Y * Y + 8 * Y + 4) / 8;
        end;
      2: 
        begin
          G[1] := (8 * X - 4 * X * X + Y * Y + 1) / 8;
          if Meth = 2 then
            X := G[1];
          G[2] := (2 * X - X * X + 4 * Y - Y * Y + 3) / 4;
        end;
      3: 
        begin
          G[1] := (2 * X - X * X + Y) / 2;
          if Meth = 2 then
            X := G[1];
          G[2] := (2 * X - X * X + 8) / 9 + (4 * Y - Y * Y) / 4;
        end;
      4: 
        begin
          G[1] := (4 * X - X * X * X + Y) / 4;
          if Meth = 2 then
            X := G[1];
          G[2] := -X * X / 9 + (4 * Y - Y * Y) / 4 + 1;
        end;
      5: 
        begin
          G[1] := (4 * X - X * X + Y + 3) / 4;
          if Meth = 2 then
            X := G[1];
          G[2] := (3 - X * Y + 2 * Y) / 2;
        end;
      6: 
        begin
          G[1] := (Y - X * X * X + 3 * X * X + 3 * X) / 7;
          if Meth = 2 then
            X := G[1];
          G[2] := (Y * Y + 2 * Y - X - 2) / 2;
        end;
    end;
  end;

  procedure PRINTFUN (FunType: integer);
  begin
    case FunType of
      1: 
        begin
          WRITELN('g (x,y) = (x^2 - y + 0.5)/2');
          WRITELN('      1');
          WRITELN('     g (x,y) = (- x^2 - 4 y^2 + 8y + 4)/8');
          WRITELN('      2');
        end;
      2: 
        begin
          WRITELN('g (x,y) = (8x - 4 x^2 + y^2 + 1)/8');
          WRITELN('      1');
          WRITELN('     g (x,y) = (2x - x^2 + 4y - y^2 + 3)/4');
          WRITELN('      2');
        end;
      3: 
        begin
          WRITELN('g (x,y) = (2x - x^2 + y)/2');
          WRITELN('      1');
          WRITELN('     g (x,y) = (2x - x^2 + 8)/9 + (4y - y^2)/4');
          WRITELN('      2');
        end;
      4: 
        begin
          WRITELN('g (x,y) = (4x - x^3 + y)/4');
          WRITELN('      1');
          WRITELN('     g (x,y) = - x^2/9 + (4y - y^2)/4 + 1');
          WRITELN('      2');
        end;
      5: 
        begin
          WRITELN('g (x,y) = (4x - x^2 + y + 3)/4');
          WRITELN('      1');
          WRITELN('     g (x,y) = (3 - xy + 2y)/2');
          WRITELN('      2');
        end;
      6: 
        begin
          WRITELN('g (x,y) = (y - x^3 + 3 x^2 + 3x)/7');
          WRITELN('      1');
          WRITELN('     g (x,y) = (y^2 + 2y - x - 2)/2');
          WRITELN('      2');
        end;
    end;
  end;

  procedure ITERATE (var P0, P1: VECTOR; var G0: VECTOR; Tol: real; Max: integer; var Sep: real; var Count: integer);

    label
      999;
    const
      Big = 1E10;
    var
      K: integer;
      D: real;
    function Dist (P0, P1: VECTOR): real;
    begin
      Dist := ABS(P1[1] - P0[1]) + ABS(P1[2] - P0[2]);
    end;
    function Size (P1: VECTOR): real;
    begin
      Size := ABS(P1[1]) + ABS(P1[2]);
    end;
  begin
    Sep := 1;
    K := 0;
    VP[0] := P0[1];  {Store an array of answers}
    VQ[0] := P0[2];
    P1 := P0;   {vector replacement}
    while (K < Max) and (Sep > Tol) do
      begin
        P0 := P1;   {vector replacement}
        K := K + 1;
        FUN(P0, G0);
        P1 := G0;
        VP[K] := P1[1];  {Store an array of answers}
        VQ[K] := P1[2];
        Sep := Dist(P0, P1);
        D := Size(P1);
        if D > Big then
          goto 999;
      end;
999:
    Count := K;
  end;                                     {End of Iteration Procedure}

  procedure STRATEGY (var Meth: integer);
  begin
    CLRSCR;
    WRITELN('          You have a choice of two methods:');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('          < 1 >   Fixed point iteration');
    WRITELN;
    WRITELN;
    WRITELN('          < 2 >   Seidel iteration');
    WRITELN;
    WRITELN;
    WRITELN;
    Mess := '          SELECT the method  < 1 or 2 >  ?   ';
    Meth := 2;
    WRITE(Mess);
    READLN(Meth);
    if Meth < 1 then
      Meth := 1;
    if Meth > 1 then
      Meth := 2;
    CLRSCR;
  end;                                  {End of procedure STRATEGY}

  procedure INPUTTOL (var Meth: integer; var Tol: real);
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('          Starting with (p ,q ) a sequences of points {(p ,q } is generated');
    WRITELN('                          0  0                           k  k');
    WRITELN;
    WRITE('     which converges to the solution. ');
    case Meth of
      1: 
        begin
          WRITELN('Fixed point iteration uses the formulas:');
          WRITELN;
          WRITELN('                         p     =  g (p ,q )');
          WRITELN('                          k+1      1  k  k ');
          WRITELN;
          WRITELN('                         q     =  g (p ,q )');
          WRITELN('                          k+1      2  k  k ');
        end;
      2: 
        begin
          WRITELN('Seidel iteration uses the formulas:');
          WRITELN;
          WRITELN('                         p     =  g (p   ,q )');
          WRITELN('                          k+1      1  k+1  k ');
          WRITELN;
          WRITELN('                         q     =  g (p ,q )');
          WRITELN('                          k+1      2  k  k ');
        end;
    end;
    WRITELN;
    WRITELN('     Termination occurs when');
    WRITELN;
    WRITELN;
    WRITELN('                        |p - p   | + |q  - q   |  <  Tol.');
    WRITELN('                          N   N-1      N    N-1');
    WRITELN;
    WRITELN;
    MESS := '                         Now  ENTER  the  value      Tol = ';
    Tol := Epsilon;
    WRITE(Mess);
    READLN(Tol);
    Tol := ABS(Tol);
    if Tol < Epsilon then
      Tol := Epsilon;
  end;                                 {End of procedure INPUTTOL}

  procedure MESSAGE (var Meth: integer; var Tol: real);
  begin
    CLRSCR;
    WRITELN('              ITERATIVE METHODS FOR NONLINEAR SYSTEMS');
    WRITELN;
    WRITELN;
    WRITELN('    Iterative methods are used for the solution of the nonlinear system');
    WRITELN;
    WRITELN;
    WRITELN('                         x  =  g (x,y)');
    WRITELN('                                1');
    WRITELN;
    WRITELN('                         y  =  g (x,y).');
    WRITELN('                                2');
    WRITELN;
    WRITELN;
    WRITELN('     The functions g (x,y) and g (x,y) must satisfy the Jacobian conditions ');
    WRITELN('                    1           2');
    WRITELN;
    WRITELN('     in order to guarantee that the iteration converges.');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN;
    WRITE('                        Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    STRATEGY(Meth);
    INPUTTOL(Meth, Tol);
  end;                                  {End of procedure MESSAGE}

  procedure GETFUN (var FunType: integer);
    var
      K: integer;
  begin
    CLRSCR;
    for K := 1 to FunMax do
      begin
        WRITE('<', K : 1, '>  ');
        PRINTFUN(K);
      end;
    Mess := '                SELECT your system of functions < 1 - 6 >  ';
    FunType := 1;
    WRITE(Mess);
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > FunMax then
      FunType := FunMax;
    CLRSCR;
  end;

  procedure GETPOINT (var FunType: integer; var Pstart, P0: VECTOR);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('     You chose the system of equations:');
    WRITELN;
    WRITELN;
    WRITE('     ');
    PRINTFUN(FunType);
    WRITELN('     ENTER the initial starting point (p ,q )');
    WRITELN('                                        0  0 ');
    Mess := '                                       p0 = ';
    P0[1] := 0;
    WRITE(Mess);
    READLN(P0[1]);
    Mess := '                                       q0 = ';
    P0[2] := 0;
    WRITE(Mess);
    READLN(P0[2]);
    WRITELN;
    Pstart := P0;   {vector replacement}
  end;

  procedure RESULTS (Pstart, P0, P1: VECTOR; Tol, Sep: real; Count: integer);

    var
      K: integer;
  begin
    K := Count;
    CLRSCR;
    WRITELN;
    WRITELN;
    case Meth of
      1: 
        WRITELN('Fixed point iteration was used to find the fixed point of the');
      2: 
        WRITELN('Seidel iteration was used to find the fixed point of the');
    end;
    WRITELN;
    WRITELN('non-linear system  x = g (x,y),  y = g (x,y)  where:');
    WRITELN('                        1             2');
    WRITE('     ');
    PRINTFUN(FunType);
    WRITELN('Starting with  p =', Pstart[1] : 15 : 7, '   and   q =', Pstart[2] : 15 : 7);
    WRITELN('                0                          0');
    WRITELN;
    if Sep < Tol then
      begin
        case Meth of
          1: 
            WRITELN('After ', Count : 2, ' iterations the fixed point method was');
          2: 
            WRITELN('After ', Count : 2, ' iterations Seidel`s method was');
        end;
        WRITELN;
        WRITELN('successful and found a solution point (P,Q).');
        WRITELN;
        WRITELN('               P =', P1[1] : 15 : 7, '         Q =', P1[2] : 15 : 7);
        WRITELN;
        WRITE('              DP =', P1[1] - P0[1] : 15 : 7, '        DQ =', P1[2] - P0[2] : 15 : 7);
      end
    else
      begin
        case Meth of
          1: 
            WRITELN('Fixed point iteration did NOT converge.');
          2: 
            WRITELN('Seidel iteration did NOT converge.');
        end;
        WRITELN;
        WRITELN('The status after ', Count : 2, ' iterations is:');
        WRITELN;
        WRITELN('           P(', K : 2, ') =', P1[1] : 15 : 7, '     Q(', K : 2, ') =', P1[2] : 15 : 7);
        WRITELN;
        WRITE('              DP =', P1[1] - P0[1] : 15 : 7, '        DQ =', P1[2] - P0[2] : 15 : 7);
      end;
  end;

  procedure PRINTAPPROXS;
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('           k             p                      q  ');
    WRITELN('                          k                      k ');
    WRITELN('         ---------------------------------------------------');
    WRITELN;
    for K := 0 to Count do
      begin
        WRITELN('          ', K : 2, '     ', VP[K] : 15 : 7, '     ', VQ[K] : 15 : 7);
        WRITELN;
        if K mod 11 = 9 then
          begin
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
          end;
      end;
    WRITELN;
    WRITE('          Press the <ENTER> key.  ');
    READLN(Ans);
  end;

begin                                            {Begin Main Program}
  Stat := Working;
  FunType := 1;
  Meth := 2;
  Tol := Epsilon;
  MESSAGE(Meth, Tol);
  while (Stat = Working) or (Stat = More) do
    begin
      if Stat = More then
        begin
          WRITELN;
          WRITE('Do you want to try  a different strategy ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            STRATEGY(Meth);
        end;
      GETFUN(FunType);
      State := Computing;
      while (State = Computing) or (State = More) do
        begin
          GETPOINT(FunType, Pstart, P0);
          ITERATE(P0, P1, G0, Tol, Max, Sep, Count);
          RESULTS(Pstart, P0, P1, Tol, Sep, Count);
          WRITELN;
          WRITELN;
          WRITE('Want  to see  all  of the approximations ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            PRINTAPPROXS;
          WRITELN;
          WRITE('Want to try  a different  starting point ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            begin
              CLRSCR;
              State := More;
            end;
        end;
      WRITELN;
      WRITE('Want to try a different nonlinear system ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done;
      if (Ans = 'Y') or (Ans = 'y') then
        Stat := More;
    end;
end.                                               {End Main Program}

